const { RSI } = require('technicalindicators');
const config = require('../config');
const { logger } = require('../utils/logger');

class RSIIndicator {
  constructor() {
    this.period = config.rsi.period;
    this.overbought = config.rsi.overbought;
    this.oversold = config.rsi.oversold;
    this.priceHistory = {}; // { symbol: [prices] }
  }

  // Agregar precio al historial
  addPrice(symbol, price) {
    if (!this.priceHistory[symbol]) {
      this.priceHistory[symbol] = [];
    }

    this.priceHistory[symbol].push(parseFloat(price));

    // Mantener solo los últimos 100 precios para no saturar memoria
    if (this.priceHistory[symbol].length > 100) {
      this.priceHistory[symbol].shift();
    }
  }

  // Calcular RSI actual
  calculate(symbol) {
    const prices = this.priceHistory[symbol];

    if (!prices || prices.length < this.period + 1) {
      logger.debug(`Insuficientes datos para RSI de ${symbol} (${prices?.length || 0}/${this.period + 1})`);
      return null;
    }

    try {
      const rsiValues = RSI.calculate({
        values: prices,
        period: this.period
      });

      const currentRSI = rsiValues[rsiValues.length - 1];
      logger.debug(`RSI de ${symbol}: ${currentRSI?.toFixed(2)}`);
      
      return currentRSI;
    } catch (error) {
      logger.error(`Error calculando RSI para ${symbol}: ${error.message}`);
      return null;
    }
  }

  // Determinar señal de trading basada en RSI
  getSignal(symbol) {
    const rsi = this.calculate(symbol);

    if (rsi === null) {
      return { signal: 'WAIT', rsi: null, reason: 'Datos insuficientes' };
    }

    // Sobreventa - señal de compra
    if (rsi < this.oversold) {
      return { 
        signal: 'BUY', 
        rsi, 
        reason: `RSI ${rsi.toFixed(2)} < ${this.oversold} (sobreventa)` 
      };
    }

    // Sobrecompra - señal de venta
    if (rsi > this.overbought) {
      return { 
        signal: 'SELL', 
        rsi, 
        reason: `RSI ${rsi.toFixed(2)} > ${this.overbought} (sobrecompra)` 
      };
    }

    // Zona neutral
    return { 
      signal: 'NEUTRAL', 
      rsi, 
      reason: `RSI ${rsi.toFixed(2)} en zona neutral` 
    };
  }

  // Verificar si RSI permite una operación
  allowsTrade(symbol, side) {
    const rsi = this.calculate(symbol);

    if (rsi === null) return false;

    // Para compras, evitar si está en sobrecompra
    if (side === 'Buy' && rsi > this.overbought) {
      logger.debug(`RSI alto (${rsi.toFixed(2)}), evitando compra de ${symbol}`);
      return false;
    }

    // Para ventas, evitar si está en sobreventa
    if (side === 'Sell' && rsi < this.oversold) {
      logger.debug(`RSI bajo (${rsi.toFixed(2)}), evitando venta de ${symbol}`);
      return false;
    }

    return true;
  }

  // Obtener estado del RSI para todos los símbolos
  getAllRSI() {
    const result = {};
    
    for (const symbol in this.priceHistory) {
      const rsi = this.calculate(symbol);
      if (rsi !== null) {
        result[symbol] = {
          rsi: rsi.toFixed(2),
          status: this.getStatus(rsi)
        };
      }
    }

    return result;
  }

  // Obtener estado descriptivo del RSI
  getStatus(rsi) {
    if (rsi < this.oversold) return 'SOBREVENTA';
    if (rsi > this.overbought) return 'SOBRECOMPRA';
    if (rsi >= 45 && rsi <= 55) return 'NEUTRAL';
    if (rsi < 45) return 'BAJISTA';
    return 'ALCISTA';
  }

  // Limpiar historial de un símbolo
  clearHistory(symbol) {
    if (this.priceHistory[symbol]) {
      delete this.priceHistory[symbol];
      logger.debug(`Historial de ${symbol} limpiado`);
    }
  }

  // Limpiar todo el historial
  clearAllHistory() {
    this.priceHistory = {};
    logger.info('Todo el historial de RSI limpiado');
  }
}

module.exports = new RSIIndicator();
