const winston = require('winston');
const path = require('path');
const config = require('../config');

// Formato personalizado
const customFormat = winston.format.combine(
  winston.format.timestamp({ format: 'YYYY-MM-DD HH:mm:ss' }),
  winston.format.errors({ stack: true }),
  winston.format.printf(({ timestamp, level, message, stack }) => {
    const logMessage = `${timestamp} [${level.toUpperCase()}]: ${message}`;
    return stack ? `${logMessage}\n${stack}` : logMessage;
  })
);

// Configuración del logger
const logger = winston.createLogger({
  level: config.system.logLevel,
  format: customFormat,
  transports: [
    // Archivo de logs generales
    new winston.transports.File({
      filename: path.join(__dirname, '../logs/bot.log'),
      maxsize: 5242880, // 5MB
      maxFiles: 5
    }),
    // Archivo solo para errores
    new winston.transports.File({
      filename: path.join(__dirname, '../logs/error.log'),
      level: 'error',
      maxsize: 5242880,
      maxFiles: 5
    }),
    // Consola con colores
    new winston.transports.Console({
      format: winston.format.combine(
        winston.format.colorize(),
        winston.format.timestamp({ format: 'HH:mm:ss' }),
        winston.format.printf(({ timestamp, level, message }) => {
          return `${timestamp} ${level}: ${message}`;
        })
      )
    })
  ]
});

// Logger especializado para trades
const tradeLogger = winston.createLogger({
  level: 'info',
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.json()
  ),
  transports: [
    new winston.transports.File({
      filename: path.join(__dirname, '../logs/trades.json'),
      maxsize: 10485760, // 10MB
      maxFiles: 10
    })
  ]
});

module.exports = { logger, tradeLogger };
