const { RestClientV5 } = require('bybit-api');
const config = require('./config');
const { logger } = require('./utils/logger');
const gridStrategy = require('./strategies/gridTrading');
const rsiIndicator = require('./strategies/rsiIndicator');
const riskManager = require('./utils/risk');

class Backtester {
  constructor() {
    this.client = new RestClientV5({
      key: config.bybit.apiKey,
      secret: config.bybit.apiSecret,
      testnet: config.bybit.testnet
    });
    
    this.results = {
      trades: [],
      totalTrades: 0,
      wins: 0,
      losses: 0,
      totalProfit: 0,
      maxDrawdown: 0,
      winRate: 0,
      profitFactor: 0,
      sharpeRatio: 0
    };
  }

  // Obtener datos históricos
  async getHistoricalData(symbol, interval = '15', limit = 200) {
    try {
      logger.info(`📊 Obteniendo datos históricos de ${symbol}...`);
      
      const response = await this.client.getKline({
        category: config.trading.category,
        symbol: symbol,
        interval: interval, // 1, 3, 5, 15, 30, 60, 120, 240, 360, 720, D, W, M
        limit: limit
      });

      if (response.retCode === 0) {
        const candles = response.result.list.reverse(); // Del más antiguo al más reciente
        
        logger.info(`✅ ${candles.length} velas obtenidas para ${symbol}`);
        
        return candles.map(candle => ({
          timestamp: parseInt(candle[0]),
          open: parseFloat(candle[1]),
          high: parseFloat(candle[2]),
          low: parseFloat(candle[3]),
          close: parseFloat(candle[4]),
          volume: parseFloat(candle[5])
        }));
      }
      
      throw new Error('No se pudieron obtener datos históricos');
    } catch (error) {
      logger.error(`Error obteniendo datos: ${error.message}`);
      throw error;
    }
  }

  // Ejecutar backtest
  async runBacktest(symbol, days = 7) {
    try {
      logger.info(`\n${'='.repeat(60)}`);
      logger.info(`🧪 INICIANDO BACKTEST PARA ${symbol}`);
      logger.info(`📅 Período: ${days} días`);
      logger.info(`💰 Capital inicial: $${config.trading.initialCapital}`);
      logger.info(`${'='.repeat(60)}\n`);

      // Calcular límite de velas (96 velas de 15min = 1 día)
      const candlesPerDay = 96;
      const limit = Math.min(days * candlesPerDay, 200);

      // Obtener datos históricos
      const candles = await this.getHistoricalData(symbol, '15', limit);
      
      // Variables de simulación
      let balance = config.trading.initialCapital;
      let position = null;
      let trades = [];
      let equity = [balance];
      let maxEquity = balance;
      let maxDrawdown = 0;

      // Inicializar grid con primer precio
      await gridStrategy.initializeGrid(symbol, candles[0].close);

      // Alimentar RSI con precios iniciales
      for (let i = 0; i < Math.min(20, candles.length); i++) {
        rsiIndicator.addPrice(symbol, candles[i].close);
      }

      logger.info(`⚙️  Procesando ${candles.length} velas...\n`);

      // Simular trading
      for (let i = 20; i < candles.length; i++) {
        const candle = candles[i];
        const price = candle.close;

        // Actualizar RSI
        rsiIndicator.addPrice(symbol, price);

        // Si tenemos posición abierta, verificar stop loss y take profit
        if (position) {
          const profit = (price - position.entryPrice) * position.qty;
          const profitPercent = (profit / (position.entryPrice * position.qty)) * 100;

          // Stop loss
          if (profitPercent <= -config.trading.stopLossPercent) {
            balance += (price * position.qty);
            
            const trade = {
              symbol,
              side: 'Sell',
              entryPrice: position.entryPrice,
              exitPrice: price,
              qty: position.qty,
              profit,
              profitPercent,
              reason: 'Stop Loss',
              timestamp: candle.timestamp
            };

            trades.push(trade);
            logger.info(`🛑 Stop Loss: ${symbol} @ $${price.toFixed(2)} | P/L: ${profit.toFixed(2)} (${profitPercent.toFixed(2)}%)`);
            
            position = null;
            gridStrategy.closePosition(symbol);
            continue;
          }

          // Take profit
          if (profitPercent >= config.trading.takeProfitPercent) {
            balance += (price * position.qty);
            
            const trade = {
              symbol,
              side: 'Sell',
              entryPrice: position.entryPrice,
              exitPrice: price,
              qty: position.qty,
              profit,
              profitPercent,
              reason: 'Take Profit',
              timestamp: candle.timestamp
            };

            trades.push(trade);
            logger.info(`🎯 Take Profit: ${symbol} @ $${price.toFixed(2)} | P/L: +${profit.toFixed(2)} (+${profitPercent.toFixed(2)}%)`);
            
            position = null;
            gridStrategy.closePosition(symbol);
            continue;
          }
        }

        // Verificar señal de grid
        const gridSignal = gridStrategy.getSignal(symbol, price, balance);
        
        if (gridSignal && !position) {
          // Verificar RSI
          const rsiAllows = rsiIndicator.allowsTrade(symbol, gridSignal.side);
          
          if (rsiAllows && gridSignal.side === 'Buy') {
            const posSize = riskManager.calculatePositionSize(balance, price);
            const cost = price * posSize;

            if (cost <= balance) {
              balance -= cost;
              position = {
                entryPrice: price,
                qty: posSize,
                timestamp: candle.timestamp
              };

              gridStrategy.openPosition(symbol, 'Buy', price, posSize);
              logger.info(`✅ Compra: ${symbol} @ $${price.toFixed(2)} | Qty: ${posSize.toFixed(6)}`);
            }
          }
        }

        // Actualizar equity y drawdown
        const currentEquity = balance + (position ? position.qty * price : 0);
        equity.push(currentEquity);
        
        if (currentEquity > maxEquity) {
          maxEquity = currentEquity;
        }
        
        const drawdown = ((maxEquity - currentEquity) / maxEquity) * 100;
        if (drawdown > maxDrawdown) {
          maxDrawdown = drawdown;
        }
      }

      // Cerrar posición abierta al final
      if (position) {
        const finalPrice = candles[candles.length - 1].close;
        balance += (finalPrice * position.qty);
        
        const profit = (finalPrice - position.entryPrice) * position.qty;
        const profitPercent = (profit / (position.entryPrice * position.qty)) * 100;

        trades.push({
          symbol,
          side: 'Sell',
          entryPrice: position.entryPrice,
          exitPrice: finalPrice,
          qty: position.qty,
          profit,
          profitPercent,
          reason: 'Final del backtest',
          timestamp: candles[candles.length - 1].timestamp
        });
      }

      // Calcular métricas
      const results = this.calculateMetrics(trades, balance, equity);
      
      // Mostrar resultados
      this.displayResults(symbol, results, balance, days);

      return results;

    } catch (error) {
      logger.error(`Error en backtest: ${error.message}`);
      throw error;
    }
  }

  // Calcular métricas de rendimiento
  calculateMetrics(trades, finalBalance, equity) {
    const totalTrades = trades.length;
    const wins = trades.filter(t => t.profit > 0).length;
    const losses = trades.filter(t => t.profit < 0).length;
    
    const totalProfit = trades.reduce((sum, t) => sum + t.profit, 0);
    const winRate = totalTrades > 0 ? (wins / totalTrades) * 100 : 0;

    const grossProfit = trades.filter(t => t.profit > 0).reduce((sum, t) => sum + t.profit, 0);
    const grossLoss = Math.abs(trades.filter(t => t.profit < 0).reduce((sum, t) => sum + t.profit, 0));
    const profitFactor = grossLoss > 0 ? grossProfit / grossLoss : 0;

    // Sharpe Ratio (simplificado)
    const returns = [];
    for (let i = 1; i < equity.length; i++) {
      returns.push((equity[i] - equity[i - 1]) / equity[i - 1]);
    }
    
    const avgReturn = returns.reduce((a, b) => a + b, 0) / returns.length;
    const variance = returns.reduce((sum, r) => sum + Math.pow(r - avgReturn, 2), 0) / returns.length;
    const stdDev = Math.sqrt(variance);
    const sharpeRatio = stdDev > 0 ? (avgReturn / stdDev) * Math.sqrt(252) : 0;

    return {
      trades,
      totalTrades,
      wins,
      losses,
      totalProfit,
      winRate,
      profitFactor,
      sharpeRatio,
      finalBalance,
      roi: ((finalBalance - config.trading.initialCapital) / config.trading.initialCapital) * 100
    };
  }

  // Mostrar resultados del backtest
  displayResults(symbol, results, finalBalance, days) {
    logger.info(`\n${'='.repeat(60)}`);
    logger.info(`📈 RESULTADOS DEL BACKTEST - ${symbol}`);
    logger.info(`${'='.repeat(60)}\n`);

    logger.info(`💰 Capital inicial: $${config.trading.initialCapital.toFixed(2)}`);
    logger.info(`💵 Capital final: $${finalBalance.toFixed(2)}`);
    logger.info(`📊 Ganancia/Pérdida: ${results.totalProfit >= 0 ? '+' : ''}$${results.totalProfit.toFixed(2)} (${results.roi >= 0 ? '+' : ''}${results.roi.toFixed(2)}%)`);
    
    logger.info(`\n📈 ESTADÍSTICAS:`);
    logger.info(`   Total de trades: ${results.totalTrades}`);
    logger.info(`   ✅ Ganadas: ${results.wins}`);
    logger.info(`   ❌ Perdidas: ${results.losses}`);
    logger.info(`   🎯 Win rate: ${results.winRate.toFixed(2)}%`);
    logger.info(`   💪 Profit factor: ${results.profitFactor.toFixed(2)}`);
    logger.info(`   📊 Sharpe ratio: ${results.sharpeRatio.toFixed(2)}`);

    if (results.trades.length > 0) {
      const profits = results.trades.map(t => t.profit);
      const bestTrade = Math.max(...profits);
      const worstTrade = Math.min(...profits);
      const avgTrade = profits.reduce((a, b) => a + b, 0) / profits.length;

      logger.info(`\n🏆 MEJORES Y PEORES:`);
      logger.info(`   Mejor trade: +$${bestTrade.toFixed(2)}`);
      logger.info(`   Peor trade: $${worstTrade.toFixed(2)}`);
      logger.info(`   Promedio: ${avgTrade >= 0 ? '+' : ''}$${avgTrade.toFixed(2)}`);
    }

    logger.info(`\n📅 Rendimiento diario promedio: ${(results.roi / days).toFixed(2)}%`);
    logger.info(`📅 Proyección mensual: ${(results.roi / days * 30).toFixed(2)}%`);

    logger.info(`\n${'='.repeat(60)}\n`);
  }

  // Ejecutar backtest para todos los pares
  async runAllBacktests(days = 7) {
    const allResults = {};

    for (const symbol of config.trading.pairs) {
      try {
        const results = await this.runBacktest(symbol, days);
        allResults[symbol] = results;
        
        // Esperar un poco entre backtests
        await new Promise(resolve => setTimeout(resolve, 2000));
      } catch (error) {
        logger.error(`Error en backtest de ${symbol}: ${error.message}`);
      }
    }

    // Mostrar resumen comparativo
    this.displayComparison(allResults);

    return allResults;
  }

  // Mostrar comparación entre pares
  displayComparison(results) {
    logger.info(`\n${'='.repeat(60)}`);
    logger.info(`📊 COMPARACIÓN DE PARES`);
    logger.info(`${'='.repeat(60)}\n`);

    const sorted = Object.entries(results)
      .sort((a, b) => b[1].roi - a[1].roi);

    sorted.forEach(([symbol, res], index) => {
      logger.info(`${index + 1}. ${symbol}:`);
      logger.info(`   ROI: ${res.roi >= 0 ? '+' : ''}${res.roi.toFixed(2)}%`);
      logger.info(`   Win Rate: ${res.winRate.toFixed(2)}%`);
      logger.info(`   Trades: ${res.totalTrades}`);
      logger.info('');
    });

    logger.info(`${'='.repeat(60)}\n`);
  }
}

// Ejecutar backtest si se llama directamente
if (require.main === module) {
  const backtester = new Backtester();
  
  // Backtest de 7 días para todos los pares
  backtester.runAllBacktests(7)
    .then(() => {
      logger.info('✅ Backtest completado');
      process.exit(0);
    })
    .catch(error => {
      logger.error(`Error: ${error.message}`);
      process.exit(1);
    });
}

module.exports = Backtester;
