const config = require('../config');
const { logger } = require('./logger');

class RiskManager {
  constructor() {
    this.dailyLosses = 0;
    this.consecutiveLosses = 0;
    this.tradesToday = 0;
    this.lastResetDate = new Date().toDateString();
    this.isPaused = false;
    this.pauseUntil = null;
  }

  // Calcular tamaño de posición basado en capital y riesgo
  calculatePositionSize(balance, price) {
    const riskAmount = balance * (config.trading.riskPerTrade / 100);
    const positionSize = riskAmount / price;
    
    logger.debug(`Position size calculado: ${positionSize} (Risk: $${riskAmount.toFixed(2)})`);
    return positionSize;
  }

  // Calcular precio de stop loss
  calculateStopLoss(entryPrice, side) {
    const percent = config.trading.stopLossPercent / 100;
    
    if (side === 'Buy') {
      return entryPrice * (1 - percent);
    } else {
      return entryPrice * (1 + percent);
    }
  }

  // Calcular precio de take profit
  calculateTakeProfit(entryPrice, side) {
    const percent = config.trading.takeProfitPercent / 100;
    
    if (side === 'Buy') {
      return entryPrice * (1 + percent);
    } else {
      return entryPrice * (1 - percent);
    }
  }

  // Verificar si se puede abrir una nueva operación
  canOpenTrade(balance) {
    // Resetear contadores diarios
    const today = new Date().toDateString();
    if (today !== this.lastResetDate) {
      this.dailyLosses = 0;
      this.tradesToday = 0;
      this.lastResetDate = today;
      logger.info('📅 Contadores diarios reseteados');
    }

    // Verificar si está en pausa
    if (this.isPaused) {
      const now = Date.now();
      if (this.pauseUntil && now < this.pauseUntil) {
        const minutesLeft = Math.ceil((this.pauseUntil - now) / 60000);
        logger.debug(`Bot en pausa. Quedan ${minutesLeft} minutos`);
        return { allowed: false, reason: `Bot en pausa por ${minutesLeft} minutos` };
      } else {
        this.resumeTrading();
      }
    }

    // Verificar pérdida máxima diaria
    const maxDailyLoss = config.trading.initialCapital * (config.trading.maxDailyLoss / 100);
    if (this.dailyLosses >= maxDailyLoss) {
      return { 
        allowed: false, 
        reason: `Pérdida diaria máxima alcanzada: $${this.dailyLosses.toFixed(2)}` 
      };
    }

    // Verificar pérdidas consecutivas (pausa temporal)
    if (this.consecutiveLosses >= 5) {
      this.pauseTrading(30); // Pausa por 30 minutos
      return { 
        allowed: false, 
        reason: '5 pérdidas consecutivas. Pausa de 30 minutos' 
      };
    }

    // Verificar que haya suficiente balance
    const minBalance = config.trading.initialCapital * 0.1; // 10% del capital inicial
    if (balance < minBalance) {
      return { 
        allowed: false, 
        reason: `Balance muy bajo: $${balance.toFixed(2)}` 
      };
    }

    return { allowed: true };
  }

  // Registrar resultado de un trade
  recordTrade(profit) {
    this.tradesToday++;

    if (profit < 0) {
      this.dailyLosses += Math.abs(profit);
      this.consecutiveLosses++;
      logger.debug(`Pérdida registrada: $${profit.toFixed(2)} (Consecutivas: ${this.consecutiveLosses})`);
    } else {
      this.consecutiveLosses = 0;
      logger.debug(`Ganancia registrada: $${profit.toFixed(2)}`);
    }
  }

  // Pausar trading
  pauseTrading(minutes) {
    this.isPaused = true;
    this.pauseUntil = Date.now() + (minutes * 60 * 1000);
    logger.warn(`⏸️  Trading pausado por ${minutes} minutos`);
  }

  // Reanudar trading
  resumeTrading() {
    this.isPaused = false;
    this.pauseUntil = null;
    this.consecutiveLosses = 0;
    logger.info('▶️  Trading reanudado');
  }

  // Ajustar tamaño de posición según rendimiento
  adjustPositionSize(baseSize, winRate, balance) {
    let multiplier = 1;

    // Si el win rate es alto, aumentar posición
    if (winRate > 70 && balance > config.trading.initialCapital * 1.05) {
      multiplier = 1.1;
      logger.debug('📈 Aumentando tamaño de posición (win rate alto)');
    }

    // Si el win rate es bajo, reducir posición
    if (winRate < 50 || balance < config.trading.initialCapital * 0.95) {
      multiplier = 0.8;
      logger.debug('📉 Reduciendo tamaño de posición (win rate bajo)');
    }

    return baseSize * multiplier;
  }

  // Calcular volatilidad basada en rango de precios
  calculateVolatility(prices) {
    if (prices.length < 2) return 0;

    const returns = [];
    for (let i = 1; i < prices.length; i++) {
      const returnVal = (prices[i] - prices[i - 1]) / prices[i - 1];
      returns.push(returnVal);
    }

    const mean = returns.reduce((a, b) => a + b, 0) / returns.length;
    const variance = returns.reduce((sum, r) => sum + Math.pow(r - mean, 2), 0) / returns.length;
    const volatility = Math.sqrt(variance) * 100;

    return volatility;
  }

  // Obtener estadísticas de riesgo
  getRiskStats() {
    return {
      dailyLosses: this.dailyLosses,
      consecutiveLosses: this.consecutiveLosses,
      tradesToday: this.tradesToday,
      isPaused: this.isPaused,
      pauseMinutesLeft: this.pauseUntil 
        ? Math.ceil((this.pauseUntil - Date.now()) / 60000) 
        : 0
    };
  }
}

module.exports = new RiskManager();
