const TelegramBot = require('node-telegram-bot-api');
const config = require('../config');
const { logger } = require('./logger');

class TelegramNotifier {
  constructor() {
    this.bot = null;
    this.chatId = config.telegram.chatId;
    this.enabled = false;

    if (config.telegram.botToken && config.telegram.chatId) {
      try {
        this.bot = new TelegramBot(config.telegram.botToken, { polling: false });
        this.enabled = true;
        logger.info('📱 Telegram notificador inicializado');
      } catch (error) {
        logger.error(`Error inicializando Telegram: ${error.message}`);
      }
    } else {
      logger.warn('⚠️  Telegram deshabilitado: falta token o chatId');
    }
  }

  async send(message, silent = false) {
    if (!this.enabled) return false;

    try {
      await this.bot.sendMessage(this.chatId, message, {
        parse_mode: 'HTML',
        disable_notification: silent
      });
      return true;
    } catch (error) {
      logger.error(`Error enviando mensaje a Telegram: ${error.message}`);
      return false;
    }
  }

  async sendStartup(capital, pairs) {
    const message = `
🤖 <b>BOT INICIADO</b>

💰 Capital: $${capital.toFixed(2)} USDT
📊 Pares: ${pairs.join(', ')}
🎯 Estrategia: Grid Trading + RSI
🌐 Exchange: Bybit ${config.bybit.testnet ? '(Testnet)' : ''}

✅ Sistema operativo 24/7
    `;
    await this.send(message.trim());
  }

  async sendTrade(trade) {
    if (!config.notifications.sendTradeNotifications) return;

    const emoji = trade.side === 'Buy' ? '🟢' : '🔴';
    const action = trade.side === 'Buy' ? 'COMPRA' : 'VENTA';
    
    const message = `
${emoji} <b>${action} EJECUTADA</b>

📌 Par: ${trade.symbol}
💵 Precio: $${parseFloat(trade.price).toFixed(2)}
📊 Cantidad: ${parseFloat(trade.qty).toFixed(6)}
💰 Total: $${(parseFloat(trade.price) * parseFloat(trade.qty)).toFixed(2)}
    `;
    await this.send(message.trim(), true);
  }

  async sendSummary(stats) {
    const profitEmoji = stats.totalProfit >= 0 ? '📈' : '📉';
    const profitSign = stats.totalProfit >= 0 ? '+' : '';
    
    const message = `
📊 <b>RESUMEN ${stats.period}</b>

${profitEmoji} P/L: ${profitSign}$${stats.totalProfit.toFixed(2)} (${profitSign}${stats.profitPercent.toFixed(2)}%)
💰 Balance: $${stats.balance.toFixed(2)} USDT

📈 Trades: ${stats.totalTrades}
  ✅ Ganadas: ${stats.wins} (${stats.winRate.toFixed(1)}%)
  ❌ Perdidas: ${stats.losses}
  
🏆 Mejor trade: +$${stats.bestTrade.toFixed(2)}
💥 Peor trade: -$${stats.worstTrade.toFixed(2)}

🎯 Próximo resumen: ${stats.nextSummary}
    `;
    await this.send(message.trim());
  }

  async sendDailySummary(stats) {
    const profitEmoji = stats.dailyProfit >= 0 ? '🌟' : '⚠️';
    const profitSign = stats.dailyProfit >= 0 ? '+' : '';
    
    const message = `
🌙 <b>RESUMEN DIARIO</b>
${new Date().toLocaleDateString('es-MX')}

${profitEmoji} Ganancia hoy: ${profitSign}$${stats.dailyProfit.toFixed(2)} (${profitSign}${stats.dailyProfitPercent.toFixed(2)}%)
💰 Balance final: $${stats.balance.toFixed(2)} USDT

📊 Trades hoy: ${stats.dailyTrades}
  ✅ Win rate: ${stats.dailyWinRate.toFixed(1)}%
  
📈 Mejor par: ${stats.bestPair}
⏱️ Tiempo activo: ${stats.uptime}

🎯 Proyección semanal: ${profitSign}$${stats.weeklyProjection.toFixed(2)}
    `;
    await this.send(message.trim());
  }

  async sendError(error, context = '') {
    if (!config.notifications.sendErrorNotifications) return;

    const message = `
⚠️ <b>ERROR DETECTADO</b>

📍 Contexto: ${context || 'General'}
❌ Error: ${error.message}

🔄 El bot continuará operando...
    `;
    await this.send(message.trim());
  }

  async sendAlert(title, description) {
    const message = `
🚨 <b>${title}</b>

${description}
    `;
    await this.send(message.trim());
  }

  async sendStopLoss(trade) {
    const message = `
🛑 <b>STOP LOSS ACTIVADO</b>

📌 Par: ${trade.symbol}
💵 Precio entrada: $${trade.entryPrice.toFixed(2)}
💵 Precio salida: $${trade.exitPrice.toFixed(2)}
📉 Pérdida: -$${trade.loss.toFixed(2)} (${trade.lossPercent.toFixed(2)}%)

⚠️ Protección activada correctamente
    `;
    await this.send(message.trim());
  }

  async sendVolatilityAlert(symbol, volatility) {
    const message = `
⚡ <b>VOLATILIDAD ALTA DETECTADA</b>

📌 Par: ${symbol}
📊 Volatilidad: ${volatility.toFixed(2)}%

🔄 Ajustando estrategia automáticamente...
    `;
    await this.send(message.trim());
  }
}

module.exports = new TelegramNotifier();
